###
# This code is used to calculate the hydraulic properties of soils in a model based on
# percent sand, percent clay, percent organic matter, and depth to restricting layer.
# The soil properties are input into Pedotransfer functions (Saxton and Rawls, 2006; Rawls et al, 1983)
# to calculate the hydraulic properties.  Refer to Appendix A of Woolridge (2019)
# for detailed instructions on creating input files and executing the code.
#
# Created by: Douglas Woolridge
# Colorado State University
#
###

import numpy as np
import arcpy
import math
from arcpy.sa import *
from arcpy.conversion import *
arcpy.CheckOutExtension("Spatial")
arcpy.CheckOutExtension("3D")
arcpy.env.overwriteOutput = True

sand = "T:\\Projects\\jdngroup\\XXXX\\psand_0_18in" # % sand raster from SSURGO
clay = "T:\\Projects\\jdngroup\\XXXX\\pclay_0_18in" # % clay raster from SSURGO
OM = "T:\\Projects\\jdngroup\\XXXX\\OM_0_18in" # % organic matter raster from SSURGO
subs = "T:\\projects\\jdngroup\\XXXX\\20180925_Chey_Subs.shp" # sub-basin shapefile
aspect = "T:\\Projects\\jdngroup\\XXXX\\chey_nf_sf_polys.shp" # shapefile of nfs and sfs polygons
d_restr = "T:\\projects\\jdngroup\\XXXX\\d_restrwa_sdv" #depth to restricting layer raster from SSURGO
Vc = Raster("T:\\projects\\jdngroup\\XXXX\\fg_epas") # vegetation cover raster from NDVI
asp_subs = "T:\\Projects\\jdngroup\\XXXX\\20180925_Chey_Subs_asp.shp" # shapefile of subbasins separated into NF and SF slopes

 
def Pedotransferfn(sand,aspect,clay,OM,sub,d_restr,Vc,asp_subs): 

    ## set extent of sand, silt, clay, OM Rasters

    tempEnvironment0 = arcpy.env.extent
    tempEnvironment1 = arcpy.env.cellSize
    tempEnvironment2 = arcpy.env.snapRaster
    arcpy.env.extent = sub
    arcpy.env.cellSize = sub
    arcpy.env.snapRaster = sub
    sandsub = ExtractByMask(sand,sub)
    claysub = ExtractByMask(clay,sub)
    OMsub = ExtractByMask(OM,sub)

    # Convert % to fraction
    sandsub = sandsub / 100
    claysub = claysub / 100
    OMsub = OMsub / 100

    ## calculate theta_s with pedotransfer functions from Saxton and Rawls (2006); Eqs 2-3, 5
    theta_33t = -0.251*sandsub + 0.195*claysub + 0.011*OMsub + 0.006*sandsub*OMsub - 0.027*claysub*OMsub + 0.452*sandsub*claysub + 0.299
    theta_33 = theta_33t + 1.283*theta_33t**2 - 0.374*theta_33t - 0.015 # field capacity soil moisture
    theta_s_33t = 0.278*sandsub + 0.034*claysub + 0.022*OMsub - 0.018*sandsub*OMsub - 0.027*claysub*OMsub - 0.584*sandsub*claysub + 0.078
    theta_s_33 = theta_s_33t + (0.636*theta_s_33t - 0.107)
    theta_s = "T:\\Projects\\jdngroup\\XXXX\\theta_s" # change path to desire location of output
    if arcpy.Exists(theta_s):
        arcpy.Delete_management(theta_s)
    theta_s = theta_33 + theta_s_33 - 0.097*sandsub + 0.043
    theta_s.save("T:\\Projects\\jdngroup\\XXXX\\theta_s") # change path to desire location of output
    theta_33.save("T:\\Projects\\jdngroup\\XXXX\\theta_33") # change path to desire location of output

    ## calculate bare ground Ksat with pedotransfer functions from Saxton and Rawls (2006); Eqs 15, 1, 18, 16
    B = "T:\\Projects\\jdngroup\\XXXX\\B" # change path to desire location of output
    theta_1500t = -0.024*sandsub + 0.487*claysub + 0.006*OMsub + 0.005*sandsub*OMsub - 0.013*claysub*OMsub + 0.068*sandsub*claysub + 0.031
    theta_1500 = theta_1500t + (0.14*theta_1500t-0.02) # wilting point soil moisture
    theta_1500.save("T:\\Projects\\jdngroup\\XXXX\\theta_1500") # change path to desire location of output
    B = 3.817/(Ln(theta_33) - Ln(theta_1500))
    lamda = 1/B
    Ksatbare = "T:\\Projects\\jdngroup\\XXXX\\Ksatbare" # change path to desire location of output
    if arcpy.Exists(Ksatbare):
        arcpy.Delete_management(Ksatbare)
    Ksatbare = 1930*(theta_s - theta_33)**(3 - lamda)
    Ksatbare.save("T:\\Projects\\jdngroup\\XXXX\\Ksatbare") # change path to desire location of output
    B.save("T:\\Projects\\jdngroup\\XXXX\\B") # change path to desire location of output
    lamda.save("T:\\Projects\\jdngroup\\XXXX\\lamda") # change path to desire location of output

    ## calculate psi_f using Saxton and Rawls, 2006 (Eqs 14, 4) and Rawls et al, 1983 (Eq. 4)
    A = Exp(3.497 + B*Ln(theta_33))
    psif = "T:\\Projects\\jdngroup\\XXXX\\psif" # change path to desire location of output
    if arcpy.Exists(psif):
        arcpy.Delete_management(psif)
    psiet = -21.67*sandsub - 27.93*claysub - 81.97*theta_s_33 + 71.12*(sandsub*theta_s_33) + 8.29*claysub*theta_s_33 + 14.05*sandsub*claysub + 27.16
    psie = psiet + (0.02*psiet**2-0.113*psiet-0.7) # bubbling pressure in kPa from Saxton and Rawls, 2006
    psie_head = psie / 9.81 * 1000 # kPa = 1000 N/m2 = 1000 (kg*m/s2) / m2 => head = P / rho*g => h = [1000 (kg * m/s2) / m2] / [(1000 kg/m3)(9.81 m/s2)]; in meters, then multiply by 1000 to convert to mm
    
    psif = (2*lamda+3)/(2*lamda+2)*(psie_head/2) # wetting front section head in mm; Eq 4 from Rawls et al, 1983
    psif.save("T:\\Projects\\jdngroup\\XXXX\\psif") # change path to desire location of output

    ## adjust bare ground ksat based on vegetation using Figure 8 in Sabol, 2008 
    ck = (Vc*100-10)/90 + 1
    Ksat = Ksatbare*ck
    Ksathalf = Ksat/2
    Ksat.save("T:\\Projects\\jdngroup\\XXXX\\Ksat") # change path to desire location of output
    Ksathalf.save("T:\\Projects\\jdngroup\\XXXX\\Ksathalf") # change path to desire location of output
    ck.save("T:\\Projects\\jdngroup\\XXXX\\ck") # change path to desire location of output

    ## calculate maximum infiltration capacity based on Green-Ampt at representative depth
    delta = 76.2 # representative depth of 3 inches in mm 
    f = "T:\\Projects\\jdngroup\\XXXX\\maxinfil" # change path to desire location of output
    if arcpy.Exists(f):
        arcpy.Delete_management(f)
    f = Ksat * (1 + psif/delta)
    f_halfks = Ksathalf * (1 + psif/delta)
    f.save("T:\\Projects\\jdngroup\\XXXX\\maxinfil") # change path to desire location of output
    f_halfks.save("T:\\Projects\\jdngroup\\XXXX\\f_halfks") # change path to desire location of output
    
    ## calculate maximum soil storage as depth * porosity * 10 (cm to mm)
    S_max = d_restr * theta_s * 10
    S_max.save("T:\\Projects\\jdngroup\\XXXX\\S_max") # change path to desire location of output
    Si_fld = d_restr * theta_33 * 10
    Si_fld.save("T:\\Projects\\jdngroup\\XXXX\\Si_fld") # change path to desire location of output
    Si_wp = d_restr * theta_1500 * 10
    Si_wp.save("T:\\Projects\\jdngroup\\XXXX\\Si_wp") # change path to desire location of output

    ## calculate statistics of all soil properties by subbasin
    arcpy.env.workspace = "T:\\Projects\\jdngroup\\XXXX\\Soil_pedotransfer" #change the workspace path to indicate where sub-basin average tables should be saved
    S_max_table = ZonalStatisticsAsTable(asp_subs,"Name",S_max,"S_max_table_WA") # 'Soil Storage (mm)'
    Si_fld_table = ZonalStatisticsAsTable(asp_subs,"Name",Si_fld,"Si_fld_table")
    Si_wp_table = ZonalStatisticsAsTable(asp_subs,"Name",Si_wp,"Si_wp_table") # 'Tension Storage (mm)'
    f_table = ZonalStatisticsAsTable(asp_subs,"Name",f,"f_table")
    f_halfks_table = ZonalStatisticsAsTable(asp_subs,"Name",f_halfks,"f_halfks_table") # 'Max Infiltration (mm/hr)'
    psif_table = ZonalStatisticsAsTable(asp_subs,"Name",psif,"psif_table")
    Ksat_table = ZonalStatisticsAsTable(asp_subs,"Name",Ksat,"Ksat_table")
    Ksathalf_table = ZonalStatisticsAsTable(asp_subs,"Name",Ksathalf,"Ksathalf_table")
    Ksatbare_table = ZonalStatisticsAsTable(asp_subs,"Name",Ksatbare,"Ksatbare_table")
    theta_s_table = ZonalStatisticsAsTable(asp_subs,"Name",theta_s,"theta_s_table")
    theta_33_table = ZonalStatisticsAsTable(asp_subs,"Name",theta_33,"theta_fld")
    theta_1500_table = ZonalStatisticsAsTable(asp_subs,"Name",theta_1500,"theta_wp")

    arcpy.env.extent = tempEnvironment0
    arcpy.env.cellSize = tempEnvironment1
    arcpy.env.snapRaster = tempEnvironment2

    return f, Ksat, theta_s, psif

f,Ksat,theta_s,psi = Pedotransferfn(sand, aspect, clay, OM,subs,d_restr,Vc,asp_subs)
