# ##
# Tc_model.py
# Created on: 2018-06-06 09:44:42.00000
#   (generated by ArcGIS/ModelBuilder)
# Description:
# The following code provides an automated tool to develop the time-area curve required for the Clark Unit Hydrograph method.
# The code calls a user-defined function (Accum_Time_function.py) that is also included in Appendix D to accumulate travel
# times to the sub-basin outlet. Refer to the Clark UH section of Appendix A of Woolridge (2019) for detailed instructions
# on using this code.  Comments have been placed within the code to indicate where the user's action is required
#
# Created by: Douglas Woolridge
# Colorado State University
#
###

# Import arcpy module
import arcpy
# import numpy module
import numpy as np
import math
from arcpy import env
from arcpy.sa import *
import matplotlib.pyplot as plt
arcpy.CheckOutExtension("Spatial")
arcpy.CheckOutExtension("3D")
arcpy.env.overwriteOutput = True

# Local variables:
# update variable paths to correspond to the specified file
# Appendix A of Woolridge et al. (2019) provides more information on creating the input files
mannings_n = "T:\\Projects\\jdngroup\\XXXX" # grid of manning's roughness coefficient values for basin
DEM = "T:\\projects\\jdngroup\\XXXX" # grid of digital elevation model for basin
stream_extent = "T:\\projects\\jdngroup\\XXXX" # grid of stream network of basin; should match stream extents as closely as possible
asp_reclass = "T:\\projects\\jdngroup\\XXXX"   # North- and south-facing polygons developed from aspect grid; flow direction raster used to determine aspect of flat locations; east flow directions classified as north and west classified as south
fac = "T:\\projects\\jdngroup\\XXXX" # flow accumulation raster with no stream burn-in performed on DEM
fdr = "T:\\projects\\jdngroup\\XXXX" # flow direction raster with value of zero burnt into the outlet cell of each subbasin
cellsize = 9.1587504 # change value based on cell size of DEM

# define array with number of subbasins to iterate through
# change array size depending on number of sub-basins in basin
subbasin=np.arange(1,4,1, dtype=int)

# initialize subbasin counter
s=0
# define array of aspects to iterate through
aspect=['North','South']

arcpy.env.workspace="T:\\projects\\jdngroup\\XXXX"

# iterate through each subbasin in basin
for x in subbasin: 
    # initialize aspect counter
    a=0
        
    # Process: Extract DEM of each sub-basin
    tempEnvironment0 = arcpy.env.cellSize
    tempEnvironment1 = arcpy.env.snapRaster
    tempEnvironment2 = arcpy.env.extent
    arcpy.env.cellSize = DEM
    arcpy.env.snapRaster = DEM
    arcpy.env.extent = DEM
    sub = "T:\\projects\\jdngroup\\XXXX\\extent_"+str(x)+'.shp' # subbasin extent shapefile; extent should be minimized to speed up travel time calculation
    DEM_sub = "T:\\projects\\jdngroup\\XXXX\\DEM_sub"+str(x) # update path to specify location file should be saved
    if arcpy.Exists(DEM_sub):
        arcpy.Delete_management(DEM_sub)
    arcpy.gp.ExtractByMask_sa(DEM, sub, DEM_sub)
    arcpy.env.cellSize = tempEnvironment0
    arcpy.env.snapRaster = tempEnvironment1        
        
    # Process: Calculate slope grid of sub-basin
    slope = "T:\\projects\\jdngroup\\XXXX\\slope"+str(subbasin[s])
    arcpy.gp.Slope_sa(DEM_sub, slope, "PERCENT_RISE", "1")

    # Process: Replace slopes less than 0.4 with a slope of 0.4; the threshold of 0.4 was selected to maximize the lake/reservoir area replaced while minimizing overland cells replaced
    slpprcnt = "T:\\projects\\jdngroup\\XXXX\\slpprc"+str(subbasin[s]) # update path to specify location file should be saved
    if arcpy.Exists(slpprcnt):
        arcpy.Delete_management(slpprcnt)
    arcpy.gp.Con_sa(slope,0.4,slpprcnt,slope,"VALUE < 0.4") # adjust this value to obtain reasonable minimum and average flow velocities within each subbasin

    # Process: Convert percent slope to decimal slope
    slpdec = Divide(slpprcnt, 100)
    slpdec.save("T:\\projects\\jdngroup\\XXXX\\slpdec"+str(subbasin[s])) # update path to specify location file should be saved

    # Process: Extract channel slope grid from subbasin slope grid
    chanslp = "T:\\projects\\jdngroup\\XXXX\\chanslp"+str(subbasin[s]) # update path to specify location file should be saved
    arcpy.gp.ExtractByMask_sa(slpdec, stream_extent, chanslp)

    # Process: Calculate flow time per channel cell
    # calculate flow time per channel cell using manning's equation: t = d * n * 1 / (Rh^(2/3) * sqrt(s))
    tempEnvironment0 = arcpy.env.snapRaster
    tempEnvironment1 = arcpy.env.mask
    tempEnvironment2 = arcpy.env.cellSize
    tempEnvironment3 = arcpy.env.extent
    arcpy.env.snapRaster = DEM_sub
    arcpy.env.mask = chanslp
    arcpy.env.cellSize = DEM_sub
    arcpy.env.extent = DEM_sub
    arcpy.env.outputCoordinateSystem = arcpy.SpatialReference(26913)
    c1 = 1.406 #From relationship developed from Livers and Wohl raw data; other sources Montgomery and Gran: c~0.015; from Emmett, 1975 Figure 38: c=0.8
    b = 0.4733 #From relationship developed from Livers and Wohl raw data; other sources Montgomery and Gran, 2003
    c2 = 0.3116 # From relationship developed from Livers and Wohl raw data; other sources: Emmett, 1975 Figure 38
    f = 0.2047 # From relationship developed from Livers and Wohl raw data; other sources: Emmett, 1975 Figure 38
    DA = Raster(fac) * cellsize**2 * 0.000001
    d = c2 * DA**f
    d.save("T:\\projects\\jdngroup\\XXXX\\depth"+str(x))# update path to specify location file should be saved
    w = c1 * DA**b
    w.save("T:\\projects\\jdngroup\\XXXX\\width"+str(x)) # update path to specify location file should be saved
    P = w + 2*d # wetted perimeter cacluation
    A = d * w #flow area
    Rh = A / P
    Rh.save("T:\\projects\\jdngroup\\XXXX\\Rh"+str(x)) # update path to specify location file should be saved

    chantime = cellsize * Raster(mannings_n) / Rh**(2.0/3.0) / 60.0 / Raster(chanslp)**0.5
    chantime.save("T:\\projects\\jdngroup\\XXXX\\chntime"+str(subbasin[s])) # update path to specify location file should be saved
    arcpy.env.snapRaster = tempEnvironment0
    arcpy.env.mask = tempEnvironment1
    arcpy.env.cellSize = tempEnvironment2
    arcpy.env.extent = tempEnvironment3

    # Calculate flow time per overland flow cell
    # Process: calculate overland flow velocity per cell using an approximation of manning's equation: V = k * sqrt(s)
    Vel = 0.3 * slpdec**0.5
    Vel.save("T:\\projects\\jdngroup\\XXXX\\vel"+str(subbasin[s]))

    # Process: calculate overland cell travel time per cell
    overlandtime = cellsize / 60 / Vel
    overlandtime.save("T:\\projects\\jdngroup\\XXXX\\ovlndt"+str(subbasin[s]))

    # Process: combine channel and overland travel time rasters to get travel time per cell raster for subbasin
    traveltime = "T:\\projects\\jdngroup\\XXXX\\travelt"+str(subbasin[s]) # update path to specify location file should be saved
    chantime = "T:\\projects\\jdngroup\\XXXX\\chntime"+str(subbasin[s]) # update path to specify location file should be saved
    arcpy.env.workspace = "T:\\projects\\jdngroup\\XXXX\\ClarkUH" # update path to specify folder where overland travel time and channel travel time rasters are saved
    rasters=["ovlndt"+str(subbasin[s]), "chntime"+str(subbasin[s])]
    arcpy.MosaicToNewRaster_management(rasters, "T:\\projects\\jdngroup\\XXXX\\ClarkUH","travelt"+str(subbasin[s]),"T:\\projects\\jdngroup\\XXXX\\*.prj","32_BIT_FLOAT",cellsize,"1","LAST") # update two file paths within "mosaic to new raster" function; the first path should specify the folder that the rasters to be mosaiced are located in and the second should specify the location of a file with the desired projection for the output  

    # Process: burn outlet points into flow direction raster    
    outlet = "T:\Projects\jdngroup\XXXX\Sub_outlets.shp" # shapefile of subbasin outlet points; shapefile must have a field named "Id" with a value of 0
    outrast = "T:\\projects\\jdngroup\\XXXX\\outrast"
    arcpy.PointToRaster_conversion(outlet,"Id", outrast) # convert outlet point shapefile to a raster; the shapefile should have an "Id" field with a value of 0 so that the raster cells have a value of 0
    tempEnvironment0 = arcpy.env.extent 
    tempEnvironment1 = arcpy.env.cellSize
    tempEnvironment2 = arcpy.env.snapRaster
    arcpy.env.extent = DEM
    arcpy.env.cellSize = DEM
    arcpy.env.snapRaster = DEM
    arcpy.env.workspace = "T:\\projects\\jdngroup\\XXXX\\ClarkUH"
    arcpy.MosaicToNewRaster_management([fdr, outrast], "T:\\projects\\jdngroup\\XXXX\\ClarkUH","fdrout","T:\\projects\\jdngroup\\XXXX\\20180925_Chey_Subs_asp.prj","8_BIT_UNSIGNED","9.1587504","1","LAST") # merge fdr raster with outlet point raster; this creates a flow direction raster with outlet cells of value=0
    fdrout = "T:\\projects\\jdngroup\\XXXX\\fdrout" # flow direction raster with outlet cells having a value of 0
    arcpy.env.extent = tempEnvironment0
    arcpy.env.cellSize = tempEnvironment1
    arcpy.env.snapRaster = tempEnvironment2

    # Process: Call function that calculates accumulated travel time to subbasin outlet for each point in subbasin
    from Accum_Time_function import Accum_Time
    totaltime = "T:\\projects\\jdngroup\\XXXX\\totalt"+str(subbasin[s])
    totaltime = Accum_Time(DEM_sub, fdrout, traveltime, cellsize)   
    
    # Calculate and plot CDF of travel times without considering aspect
    traveltimes=arcpy.RasterToNumPyArray(totaltime)
    T=traveltimes[traveltimes>0]
    Perc_Time=np.divide(T,np.amax(T))
    Perc_Area=np.divide(np.linspace(1,len(T),len(T)),len(T))
    np.savetxt("T:\\projects\\jdngroup\\XXXX\\TA_NoAsp"+str(subbasin[s])+".csv",np.column_stack((np.sort(Perc_Time).flatten(),Perc_Area.flatten())),delimiter=',', header="Perc Time,Perc Area")   
    plt.plot(np.sort(Perc_Time),Perc_Area, linewidth=3, label='Entire Sub-basin', color='k')
    plt.title('Subbasin '+str(subbasin[s])+' Time-Area Curve', fontsize=16)
    plt.ylabel('Area (% of Total)', fontsize=14)
    plt.xlabel('Time (% of Total)', fontsize=14)
    plt.grid(linestyle='--', linewidth=.3)
    lines=[':','--']
    
    # iterate through aspects
    for y in aspect:
        
        # Process: Extract by Mask (2)
        totalt_sub_asp = "T:\\projects\\jdngroup\\XXXX\\t_tot"+str(aspect[a])+str(subbasin[s])
        if arcpy.Exists(totalt_sub_asp):
            arcpy.Delete_management(totalt_sub_asp)
        arcpy.MakeFeatureLayer_management(asp_reclass,"asp_layer")
        arcpy.SelectLayerByAttribute_management ("asp_layer", "NEW_SELECTION", """{0}='{1}'""".format("ASPECT",str(aspect[a])))
        tempEnvironment0 = arcpy.env.snapRaster
        tempEnvironment1 = arcpy.env.mask
        arcpy.env.snapRaster = DEM_sub
        arcpy.env.mask = DEM_sub
        arcpy.gp.ExtractByMask_sa(totaltime, "asp_layer", totalt_sub_asp)
        arcpy.env.snapRaster = tempEnvironment0
        arcpy.env.mask = tempEnvironment1
        # Calculate and plot CDF of travel times for each aspect
        traveltimes=arcpy.RasterToNumPyArray(totalt_sub_asp)
        T=traveltimes[traveltimes>0]
        Perc_Time=np.divide(T,np.amax(T))
        Perc_Area=np.divide(np.linspace(1,len(T),len(T)),len(T))
        np.savetxt("T:\\projects\\jdngroup\\XXXX\\TA"+str(aspect[a])+str(subbasin[s])+".csv",np.column_stack((np.sort(Perc_Time).flatten(),Perc_Area.flatten())),delimiter=',', header="Perc Time,Perc Area")
        
        plt.plot(np.sort(Perc_Time),Perc_Area, linewidth=2, label=str(aspect[a])+'-Facing Slopes', linestyle=lines[a], color='blue')      
        
        
        a+=1
    plt.legend(loc='lower right')
    plt.savefig("T:\projects\jdngroup\XXXX\Sub"+str(subbasin[s])+".png")
    plt.clf()
    s+=1
